<?php

namespace App\Http\Controllers;

use App\Models\AddressBook;
use Illuminate\Http\Request;
use App\Http\Requests\AddreesStore;
use App\Models\CountryCode;
use App\Services\AddressBookService;
use Illuminate\Support\Facades\Auth;

class AddressBookController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $query = AddressBook::with('country', 'countryCode', 'client');

        if (Auth::user()->user_type_id == 1) {
            $query->where('client_id', Auth::id());
        }

        $records = $query->get();

        return view('app.address-book.index', compact('records'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $title = 'Add Address';
        $action = 'create';
        $address = null;
        $country = CountryCode::get();
        $html = view('app.address-book.form', compact('title', 'action', 'address', 'country'))->render();

        return response()->json(['html' => $html], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AddreesStore $request, AddressBookService $addressBookService)
    {
        $client_id = $request->client_id ?? Auth::id();

        if ($client_id == 0) {
            $client_id = Auth::id();
        }

        try {
            $address = $addressBookService->store($request, $client_id);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to add address', 'error' => $e->getMessage()], 500);
        }


        return response()->json(['message' => 'Address added successfully', 'data' => $address], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(AddressBook $addressBook)
    {
        return response()->json(['message' => 'Address Book Info', 'data' => $addressBook], 200);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($addressBook)
    {
        $address = AddressBook::with('country', 'countryCode')->find($addressBook);
        $title = 'Edit Address';
        $action = 'update';
        $country = CountryCode::get();
        $html = view('app.address-book.form', compact('title', 'action', 'address', 'country'))->render();

        return response()->json(['html' => $html], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, AddressBook $addressBook)
    {
        $data = $request->all();

        $client_id = $request->client_id ?? Auth::id();
        if ($client_id == 0) {
            $data['client_id'] = Auth::id();
        }


        try {
            $addressBook->update($data);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to update address', 'error' => $e->getMessage()], 500);
        }

        return response()->json(['message' => 'Address updated successfully', 'data' => $addressBook], 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(AddressBook $addressBook)
    {
        $addressBook->delete();

        return response()->json(['message' => 'Address deleted successfully'], 200);
    }
}
